//
//  AHKPasswordWindowController.m
//  AHK3001V Address Utility
//
//  Created by FUJIDANA on Fri Apr 16 2005.
//  Copyright (c) 2005 FUJIDANA. All rights reserved.
//
//
//  PasswordController.m
//  FileUtility
//
//  Created by raktajino on Thu 2005/03/07.
//  Copyright (c) 2005 raktajino. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "AHKPasswordWindowController.h"
#import <Security/Security.h> 

const char *kServiceName = "AH-K3001V Password";	// パスワードをキー・チェーンに登録する際のサービス名
const char *kAccountName = "AH-K3001V";				// パスワードをキー・チェーンに登録する際のアカウント名

@implementation AHKPasswordWindowController

#pragma mark -
#pragma mark class method
+ (id)sharedWindowController
{
	static AHKPasswordWindowController *staticPasswordWindowController = nil;
	if (!staticPasswordWindowController) {
		staticPasswordWindowController = [[AHKPasswordWindowController alloc] init];
	}
	return staticPasswordWindowController;
}

#pragma mark -
#pragma mark initialization and deallocation

- (id)init
{
	self = [self initWithWindowNibName:@"AHKPasswordWindow"];
	if (self != nil) {
		currentPassword = nil;
	}
	return self;
}

- (void)dealloc
{
	[currentPassword release];
	
	[super dealloc];
}

#pragma mark other methods

// --- パスワードを検索する。

- (NSString *)findPassword
{
	NSString *password = nil;
	if(currentPassword != nil) {
		password = currentPassword;
	} else {
		OSStatus result;
		UInt32   passwordLength = 0;
		void    *passwordData   = nil;
		result = SecKeychainFindGenericPassword(NULL,
												strlen(kServiceName), kServiceName,
												strlen(kAccountName), kAccountName,
												&passwordLength, &passwordData, NULL);
		if(result == noErr) {
			// 見つかったパスワードをNSStringに変換する。
			password = [[[NSString alloc] initWithBytes:passwordData length:passwordLength encoding:NSUTF8StringEncoding] autorelease];
			SecKeychainItemFreeContent(NULL, passwordData);
		}
	}
	return password;
}

// --- シートを表示し、パスワードの入力を求める。

- (void)inputPasswordForWindow:(NSWindow *)parentWindow modalDelegate:(id)delegate didEndActionSelector:(SEL)selector
{
	NSDictionary *options = [[NSDictionary alloc] initWithObjectsAndKeys:[NSNumber numberWithLong:(long)selector], @"didEndActionSelector",
																		 delegate,								   @"modalDelegate",
																		 nil];
	[passwordField setStringValue:@""];
	[savePasswordButton setState:NSOffState];
	[NSApp beginSheet:[self window]
	   modalForWindow:parentWindow
		modalDelegate:self
	   didEndSelector:@selector(passwordSheetDidEnd:returnCode:contextInfo:)
		  contextInfo:options];
}

// --- シートのOKボタンがクリックされた。

- (IBAction)clickOKButton:(id)sender
{
    [NSApp endSheet:[self window] returnCode:NSOKButton];
}

// --- シートのキャンセル・ボタンがクリックされた。

- (IBAction)clickCancelButton:(id)sender
{
    [NSApp endSheet:[self window] returnCode:NSCancelButton];
}

// --- シートの入力結果を元に、パスワードをキーチェーンに登録する。

- (void)passwordSheetDidEnd:(NSWindow *)sheet returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
    [sheet orderOut:self];
	
	if(returnCode == NSOKButton) {
		// パスワードをキーチェーンに保存する。
		if([savePasswordButton state] == NSOnState) {
			// 入力されたパスワードをUTF8形式に変換する。
			NSData *newPassword = [[passwordField stringValue] dataUsingEncoding:NSUTF8StringEncoding];
			
			// パスワードがキーチェーンに登録済みかどうか調べる。
			OSStatus           result;
			UInt32             passwordLength = 0;
			void              *passwordData   = nil;
			SecKeychainItemRef itemRef        = nil;
			result = SecKeychainFindGenericPassword(NULL,
													strlen(kServiceName), kServiceName,
													strlen(kAccountName), kAccountName,
													&passwordLength, &passwordData, &itemRef);
			if(result == noErr) {
				// パスワードが登録されていたので、新しいパスワードで置き換える。
				SecKeychainItemFreeContent(NULL, passwordData);
				SecKeychainAttribute attrs[] =  {
													{ kSecAccountItemAttr, strlen(kAccountName), (char *)kAccountName },
													{ kSecServiceItemAttr, strlen(kServiceName), (char *)kServiceName }
												};
				const SecKeychainAttributeList attributes = { sizeof(attrs) / sizeof(attrs[0]), attrs };
				SecKeychainItemModifyAttributesAndData(itemRef, &attributes, [newPassword length], [newPassword bytes]);
				CFRelease(itemRef);
			}
			if(result == errSecItemNotFound) {
				// パスワードは未登録だったので、新たに登録する。
				SecKeychainAddGenericPassword(NULL,
											  strlen(kServiceName), kServiceName,
											  strlen(kAccountName), kAccountName,
											  [newPassword length], [newPassword bytes], NULL);
			}
			
			// メモリ上のパスワードを消去する。
			[currentPassword release];
			currentPassword = nil;
		}
		else {
			// パスワードをキーチェーンではなく、一時的にメモリ上へ保存する。
			[currentPassword release];
			currentPassword = [[passwordField stringValue] copy]; 
		}
		
		// パスワードの入力完了をdelageteへ通知する。
		NSDictionary *options              = [(NSDictionary*)contextInfo autorelease];
		SEL           didEndActionSelector = (SEL)[[options objectForKey:@"didEndActionSelector"] longValue];
		id            modalDelegate        = [options objectForKey:@"modalDelegate"];
		[modalDelegate performSelector:didEndActionSelector withObject:modalDelegate];
	}
}

@end
